unit Progbar;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, Menus;

type
  TProgressBar = class(TCustomControl)
  private
    { Private declarations }
    FMax :Integer;
    FMin :Integer;
    FPosition :Integer;
    FStep :Integer;
    FDivCount :Integer;
    FDivWidth :Integer;
    FDivNow :Integer;
    procedure SetMax(Value :Integer);
    procedure SetMin(Value :Integer);
    procedure SetPosition(Value :Integer);
    procedure SetDivisions;
    procedure WMSize(var Message: TWMSize); message WM_SIZE;
  protected
    { Protected declarations }
    procedure Paint; override;
  public
    { Public declarations }
    constructor Create(AOwner :TComponent); override;
    destructor Destroy; override;
    procedure StepBy(Delta :Integer);
    procedure StepIt;
  published
    { Published declarations }
    property Align;
    property Enabled;
    property Max :Integer read FMax write SetMax default 0;
    property Min :Integer read FMin write SetMin default 100;
    property ParentShowHint;
    property PopupMenu;
    property Position :Integer read FPosition write SetPosition;
    property ShowHint;
    property Step :Integer read FStep write FStep default 10;
    property TabStop;
    property Visible;
    property OnEnter;
    property OnExit;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Custom', [TProgressBar]);
end;

constructor TProgressBar.Create(AOwner :TComponent);

Begin
  inherited Create(AOwner);
  ControlStyle:=[csOpaque];
  Width:=150;
  Height:=16;
  Min:=0;
  Max:=100;
  Step:=10;
End;

destructor TProgressBar.Destroy;

Begin
  inherited Destroy;
End;

procedure TProgressBar.Paint;

Var
  d,rc :Integer;
  dr,r :TRect;

Begin
  With Canvas Do
    Begin
      r:=GetClientRect;
      Pen.Color:=clBtnShadow;
      MoveTo(r.Left,r.Bottom-1);
      LineTo(r.Left,r.Top);
      LineTo(r.Right-1,r.Top);
      Pen.Color:=clBtnHighlight;
      LineTo(r.Right-1,r.Bottom-1);
      LineTo(r.Left-1,r.Bottom-1);
      InflateRect(r,-1,-1);
      Pen.Color:=clBtnFace;
      rc:=r.Right;
      For d:=1 To FDivCount Do
        Begin
          If d>FDivNow Then
            Brush.Color:=clBtnFace
          Else
            Brush.Color:=clBtnShadow;
          r.Right:=r.Left+FDivWidth;
          dr:=r;
          If dr.Right>rc Then
            dr.Right:=rc;
          Rectangle(dr.Left,dr.Top,dr.Right,dr.Bottom);
          Inc(r.Left,FDivWidth);
        End;
    End;
End;

procedure TProgressBar.SetMax(Value :Integer);

Begin
  If (Value=FMax) Or (Value<Min) Then
    Exit;
  FMax:=Value;
  If FPosition>FMax Then
    FPosition:=FMax;
  SetDivisions;
End;

procedure TProgressBar.SetMin(Value :Integer);

Begin
  If (Value=FMin) Or (Value>Max) Then
    Exit;
  FMin:=Value;
  If FPosition<FMin Then
    FPosition:=FMin;
  SetDivisions;
End;

procedure TProgressBar.SetPosition(Value :Integer);

Begin
  If (Value=FPosition) Or (Value<Min) Or (Value>Max) Then
    Exit;
  FPosition:=Value;
  SetDivisions;
End;

procedure TProgressBar.WMSize(var Message: TWMSize);

Begin
  inherited;
  SetDivisions;
End;

procedure TProgressBar.SetDivisions;

Var
  dw,dc,dn :Integer;
  c        :Real;

Begin
  dw:=Trunc((Height-4)/1.5)+2;
  c:=(Width-4)/dw;
  dc:=Trunc(c);
  If Frac(c)<>0 Then
    Inc(dc);
  If (FMax-FMin)=0 Then
    dn:=0
  Else
    dn:=Round(((FPosition-FMin)/(FMax-FMin))*dc);
  If (dw<>FDivWidth) Or (dc<>FDivCount) Or (dn<>FDivNow) Then
    Begin
      FDivWidth:=dw;
      FDivCount:=dc;
      FDivNow:=dn;
      Refresh;
    End;
End;

procedure TProgressBar.StepBy(Delta :Integer);

Begin
  Position:=Position+Delta;
End;

procedure TProgressBar.StepIt;

Begin
  Position:=Position+FStep;
End;

end.
